/*
 * pgpTokenLib.h -- Structures for accessing Cryptoki tokens
 *
 * $Id: pgpTokenLib.h,v 1.38.2.4 2001/05/15 20:51:37 ajivsov Exp $
 */

#ifndef Included_pgpTokenLib_h
#define Included_pgpTokenLib_h

#include "pgpUsuals.h"
#include "pgpKeys.h"	/* to get 'PGPPublicKeyAlgorithm' type */

PGP_BEGIN_C_DECLARATIONS

#define TCL_PRIVATE_KEY_DATA_LIST PGPByte *ID, \
	PGPByte *modulus, \
	PGPSize modulusSize, \
	PGPByte *pubExp, \
	PGPSize pubExpSize, \
	PGPByte *privExp, \
	PGPSize privExpSize, \
	PGPByte *prime1, \
	PGPSize prime1Size, \
	PGPByte *prime2, \
	PGPSize prime2Size, \
	PGPByte *exp1, \
	PGPSize exp1Size, \
	PGPByte *exp2, \
	PGPSize exp2Size, \
	PGPByte *coeff, \
	PGPSize coeffSize

#include "pgpOpaqueStructs.h"

#define MAXTOKENS 1

#if PGP_WIN32
#pragma pack(push, p11, 1)
#endif
typedef struct _PGPTokenKeyInfo {
	PGPByte		keyid[8];

	PGPBoolean	pgpData; 
	PGPUInt8	alg;	/* kPGPPublicKeyAlgorithm_RSA */

	PGPByte		opaque[2+4+20];
} PGPTokenKeyInfo;
#if PGP_WIN32
#pragma pack(pop, p11)
#endif

typedef struct _PGPTokenCertInfo {
	PGPByte keyid[8];   /* Will be the same */
	PGPByte opaque[4];
} PGPTokenCertInfo;

#if PGP_WIN32
#pragma pack(push, p11, 1)
#endif
    typedef struct _pgpTokenPubKeyStub {  /* is written to the token */
        PGPByte     flags;
        PGPUInt32   creationtime;
    } pgpTokenPubKeyStub;

#if PGP_WIN32
#pragma pack(pop, p11)
#endif

typedef struct _pgpTokenDataInfo {  
    pgpTokenPubKeyStub  pubKeyStub; 
    PGPUInt32           alg;
        /* RSA portion if alg == kPGPPublicKeyAlgorithm_RSA */
    PGPSize             exp_size;   /*in bytes*/ 
    PGPSize             mod_size;   /*in bytes*/
       /* RSA public data goes here: exponent first, next modulus */
} pgpTokenDataInfo;


struct PGPToken
{
	PGPError		(*init)			(PGPToken *tptr, PGPUInt32 whichslot,
									PGPBoolean haveslot, PGPInt32 tokenNum);
	PGPError		(*destroy)		(PGPToken *tptr);
	PGPError		(*logout)		(PGPToken *tptr);
	PGPError		(*login)		(PGPToken *tptr, char const *input,
									PGPSize length);
	PGPError		(*decrypt)		(PGPToken *tptr, PGPByte *keyID, 
									PGPByte *in, PGPSize isize,
									PGPByte *out, PGPSize osize);
	PGPError		(*sign)			(PGPToken *tptr, PGPByte *keyID,
									PGPByte *in, PGPSize isize,
									PGPByte *out, PGPSize osize);

	PGPTokenKeyInfo* (*getPrivKeyIDs)(PGPToken *tptr, PGPSize *n);
	PGPTokenKeyInfo* (*getPubKeyIDs) (PGPToken *tptr, PGPSize *n);
	PGPByte *		(*getPublicData)(PGPToken *tptr, PGPByte *keyID, 
									PGPSize *size, pgpTokenDataInfo **di);
    PGPTokenCertInfo * (*getX509Certs) (PGPToken *tptr, PGPTokenKeyInfo *keyID, 
									PGPInt32 *n );
    PGPByte *       (*getX509Cert) (PGPToken *tptr, PGPTokenCertInfo *certInfo, 
									PGPSize *size );
	PGPError		(*putPublicData)(PGPToken *tptr, PGPByte *keyID,
									PGPByte *pubKeyBlock, PGPSize size, 
									pgpTokenDataInfo *di);
	PGPError		(*putPrivate)	(PGPToken *tptr, PGPBoolean is_master, TCL_PRIVATE_KEY_DATA_LIST);

	PGPError		(*del)			(PGPToken *tptr, const PGPByte *keyID, PGPBoolean is_last);

	PGPError		(*keygen)		(PGPToken *tptr, PGPByte *ID, 
									PGPSize modulusSize, PGPBoolean genMaster, 
									PGPByte *modulus, PGPSize *pubExp);
	PGPError		(*editKeyList)  (PGPToken *tptr, PGPByte *origID,
									PGPByte *newID);
	PGPError		(*setPIN)		(PGPToken *tptr, char *oldPIN,
									PGPSize oldPINsize, char *newPIN,
									PGPSize newPINsize);
	PGPError		(*wipe)			(PGPToken *tptr);
    PGPError		(*getInfo)      (PGPToken *tptr, PGPTokenInfo *tokenInfo);
    PGPError		(*importX509)   (PGPToken *tptr, const PGPByte *keyID, 
                                    const PGPByte *userID, PGPSize userID_len, 
                                    const PGPByte *x509, PGPSize x509_len, 
                                    const PGPByte *subjDER, PGPSize subjDER_len, 
                                    const PGPByte *issuerDER, PGPSize issuerDER_len,
                                    const PGPByte *sn, PGPSize sn_len);
    PGPError        (*putKeyContainer)( PGPToken *tptr,
                                    const PGPByte *keyID, 
                                    const PGPByte *cont, PGPSize contSize );
    PGPError        (*getKeyContainer)( PGPToken *tptr,
                                    const PGPByte *keyID, 
                                    PGPByte **contOut, PGPSize *contOutSize );
	PGPBoolean logged_in;
	PGPUInt32 session;
	PGPBoolean have_session;
	PGPUInt32 slot;
	PGPBoolean have_slot;
	PGPUInt32 tokenNum1; /* token index, starting from 1. 0 is invalid value */

    PGPTokenInfo info;

	/* Caching. Should not be accessed directly */
	PGPTokenKeyInfo *privKeyIDs;
	int			nPrivKeyIDs;
	PGPTokenKeyInfo *pubKeyIDs;
	int			nPubKeyIDs;
	PGPUInt32	cachedTimeStamp;
};


/* Macros to access virtual functions */
#define pgpTokenObjDecrypt(p,k,i,il,o,ol)		(p)->decrypt(p,k,i,il,o,ol)
#define pgpTokenObjSign(p,k,i,il,o,ol)			(p)->sign(p,k,i,il,o,ol)
#define pgpTokenObjAuth(p,s,l)					(p)->login(p,s,l)
#define pgpTokenObjDeAuth(p)					(p)->logout(p)
#define pgpTokenObjGetPublicKeyData(p,k,s,di)	(p)->getPublicData(p,k,s,di)
#define pgpTokenObjGetX509Certs(p,k,n)		    (p)->getX509Certs(p,k,n)
#define pgpTokenObjGetX509Cert(p,c,s)		    (p)->getX509Cert(p,c,s)
#define pgpTokenObjPutPrivKey(p,ism,k,mb,ml,eb,el,db,dl,pb,pl,qb,ql,xb,xl,yb,yl,cb,cl) \
				(p)->putPrivate(p,ism,k,mb,ml,eb,el,db,dl,pb,pl,qb,ql,xb,xl,yb,yl,cb,cl)
#define pgpTokenObjPutPubKeyData(p,k,cb,cl,di)(p)->putPublicData(p,k,cb,cl,di)

#define pgpTokenObjDeleteKey(p,k,l)				(p)->del(p,k,l)

#define pgpTokenObjGetPrivKeyIDs(p,s)			(p)->getPrivKeyIDs(p,s)
#define pgpTokenObjGetPubKeyIDs(p,s)			(p)->getPubKeyIDs(p,s)
#define pgpTokenObjKeygen(p,k,s,gm,m,e)			(p)->keygen(p,k,s,gm,m,e)
#define pgpTokenObjEditKeyList(p,o,n)			(p)->editKeyList(p,o,n)
#define pgpTokenObjSetPIN(p,o,ol,n,nl)			(p)->setPIN(p,o,ol,n,nl)
#define pgpTokenObjWipe(p)						(p)->wipe(p)
#define pgpTokenObjGetInfo(p,i)			        (p)->getInfo(p,i)
#define pgpTokenObjImportX509(p,id,uid,uidl,x509,x509l,s,sl,i,il,sn,snl) (p)->importX509(p,id,uid,uidl,x509,x509l,s,sl,i,il,sn,snl)
#define pgpTokenObjPutKeyContainer(p,id,c,cs)   (p)->putKeyContainer(p,id,c,cs)
#define pgpTokenObjGetKeyContainer(p,id,c,cs)   (p)->getKeyContainer(p,id,c,cs)


PGPInt32		pgpAcquireAllTokens(void);
void			pgpReleaseAllTokens(void);
PGPError		pgpTokenInit(void);

PGPUInt32		pgpCountTokenObjects(void);

PGPToken*		pgpGetTokenObjectByPrivKeyID(const PGPByte *keyID);

PGPToken*	    pgpGetNthTokenObject(PGPUInt32 x);

PGPInt32	    pgpGetTokenNum(PGPToken*token);

PGP_END_C_DECLARATIONS

#endif /* Included_pgpTokenLib_h */
